using System;
using System.Runtime.InteropServices;

namespace Sony
{
	namespace NP
	{
		public class TusTss
		{
			// Structure containig retrieved TUS or TSS data.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			struct TusTssData
			{
				public int dataSize;
				IntPtr _data;
				public byte[] data			// Byte array containing the data.
				{
					get {
						byte[] bytes = new byte[dataSize];
						Marshal.Copy(_data, bytes, 0, dataSize);
						return bytes;
					}
				}
			};

			// Structure containing a retrieved TUS variable.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct TusRetrievedVariable
			{
				public Int64 lastChangedDate;
				public Int64 variable;
				public Int64 oldVariable;
				int npIDSize;
				IntPtr _ownerNpID;
				IntPtr _lastChangeAuthorNpID;
				public int hasData;

				public byte[] ownerNpID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_ownerNpID, bytes, 0, npIDSize);
						return bytes;
					}
				}

				public byte[] lastChangeAuthorNpID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_lastChangeAuthorNpID, bytes, 0, npIDSize);
						return bytes;
					}
				}
			};

			// Structure for setting a TUS variable.
			public struct TusVariable
			{
				public TusVariable(int _slotID, long _value)
				{
					slotID = _slotID;
					value = _value;
				}
				public int slotID;
				public Int64 value;
			};

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTUSIsBusy();

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTusTssGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxTusTssGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSClearVariablesToSet();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSAddVariableToSet(int slot, Int64 value);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSSetVariables();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSSetVariablesForUser(byte[] npID);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxTUSSetVariablesForVirtualUser(string onlineID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSModifyVariables();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSModifyVariablesForUser(byte[] npID);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxTUSModifyVariablesForVirtualUser(string onlineID);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSClearVariablesToGet();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSAddVariableToGet(int slot);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSRequestVariables();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSRequestVariablesForUser(byte[] npID);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxTUSRequestVariablesForVirtualUser(string onlineID);
			[DllImport("UnityNpToolkit")]
			private static extern int PrxTUSGetVariableCount();
			[DllImport("UnityNpToolkit")]
			private static extern Int64 PrxTUSGetVariableValue(int index);
			[DllImport("UnityNpToolkit")]
			private static extern Int64 PrxTUSGetVariable(int index, out TusRetrievedVariable variable);

			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSSetData(int slotID, byte[] data, int dataSize);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSRequestData(int slotID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSSetDataForUser(int slotID, byte[] data, int dataSize, byte[] npID);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxTUSSetDataForVirtualUser(int slotID, byte[] data, int dataSize, string onlineID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSRequestDataForUser(int slotID, byte[] npID);
			[DllImport("UnityNpToolkit", CharSet = CharSet.Ansi)]
			private static extern ErrorCode PrxTUSRequestDataForVirtualUser(int slotID, string onlineID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTUSGetData(out TusTssData data);

			[DllImport("UnityNpToolkit")]
			private static extern bool PrxTSSIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTSSRequestData();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTSSRequestDataFromSlot(int slot);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxTSSGetData(out TusTssData data);

			// Is the word filter busy with a request.
			public static bool IsTusBusy()
			{
				return PrxTUSIsBusy();
			}

			public static ErrorCode SetTusVariables(TusVariable[] variables)
			{
				if(variables.Length > 0)
				{
					PrxTUSClearVariablesToSet();
					for(int i=0; i<variables.Length; i++)
					{
						PrxTUSAddVariableToSet(variables[i].slotID, variables[i].value);
					}
					return PrxTUSSetVariables();
				}

				return ErrorCode.NP_OK;
			}

			public static ErrorCode SetTusVariablesForUser(byte[] npID, TusVariable[] variables)
			{
				if(variables.Length > 0)
				{
					PrxTUSClearVariablesToSet();
					for(int i=0; i<variables.Length; i++)
					{
						PrxTUSAddVariableToSet(variables[i].slotID, variables[i].value);
					}
					return PrxTUSSetVariablesForUser(npID);
				}
				
				return ErrorCode.NP_OK;
			}
			
			public static ErrorCode SetTusVariablesForVirtualUser(string onlineID, TusVariable[] variables)
			{
				if(variables.Length > 0)
				{
					PrxTUSClearVariablesToSet();
					for(int i=0; i<variables.Length; i++)
					{
						PrxTUSAddVariableToSet(variables[i].slotID, variables[i].value);
					}
					return PrxTUSSetVariablesForVirtualUser(onlineID);
				}
				
				return ErrorCode.NP_OK;
			}

			public static ErrorCode ModifyTusVariables(TusVariable[] variables)
			{
				if(variables.Length > 0)
				{
					PrxTUSClearVariablesToSet();
					for(int i=0; i<variables.Length; i++)
					{
						PrxTUSAddVariableToSet(variables[i].slotID, variables[i].value);
					}
					return PrxTUSModifyVariables();
				}
				
				return ErrorCode.NP_OK;
			}

			public static ErrorCode ModifyTusVariablesForUser(byte[] npID, TusVariable[] variables)
			{
				if(variables.Length > 0)
				{
					PrxTUSClearVariablesToSet();
					for(int i=0; i<variables.Length; i++)
					{
						PrxTUSAddVariableToSet(variables[i].slotID, variables[i].value);
					}
					return PrxTUSModifyVariablesForUser(npID);
				}
				
				return ErrorCode.NP_OK;
			}
			
			public static ErrorCode ModifyTusVariablesForVirtualUser(string onlineID, TusVariable[] variables)
			{
				if(variables.Length > 0)
				{
					PrxTUSClearVariablesToSet();
					for(int i=0; i<variables.Length; i++)
					{
						PrxTUSAddVariableToSet(variables[i].slotID, variables[i].value);
					}
					return PrxTUSModifyVariablesForVirtualUser(onlineID);
				}
				
				return ErrorCode.NP_OK;
			}

			public static ErrorCode RequestTusVariables(int[] slotIDs)
			{
				if(slotIDs.Length > 0)
				{
					PrxTUSClearVariablesToGet();
					for(int i=0; i<slotIDs.Length; i++)
					{
						PrxTUSAddVariableToGet(slotIDs[i]);
					}
					return PrxTUSRequestVariables();
				}
				
				return ErrorCode.NP_OK;
			}
			
			public static ErrorCode RequestTusVariablesForUser(byte[] npID, int[] slotIDs)
			{
				if(slotIDs.Length > 0)
				{
					PrxTUSClearVariablesToGet();
					for(int i=0; i<slotIDs.Length; i++)
					{
						PrxTUSAddVariableToGet(slotIDs[i]);
					}
					return PrxTUSRequestVariablesForUser(npID);
				}
				
				return ErrorCode.NP_OK;
			}
			
			public static ErrorCode RequestTusVariablesForVirtualUser(string onlineID, int[] slotIDs)
			{
				if(slotIDs.Length > 0)
				{
					PrxTUSClearVariablesToGet();
					for(int i=0; i<slotIDs.Length; i++)
					{
						PrxTUSAddVariableToGet(slotIDs[i]);
					}
					return PrxTUSRequestVariablesForVirtualUser(onlineID);
				}
				
				return ErrorCode.NP_OK;
			}

			public static Int64[] GetTusVariablesValue()
			{
				int count = PrxTUSGetVariableCount();
				Int64[] variables = new Int64[count];
				for(int i=0; i<count; i++)
				{
					variables[i] = PrxTUSGetVariableValue(i);
				}
				return variables;
			}

			public static TusRetrievedVariable[] GetTusVariables()
			{
				int count = PrxTUSGetVariableCount();
				TusRetrievedVariable[] variables = new TusRetrievedVariable[count];
				for(int i=0; i<count; i++)
				{
					variables[i] = new TusRetrievedVariable();
					PrxTUSGetVariable(i, out variables[i]);
				}
				return variables;
			}

			public static ErrorCode SetTusData(int slotID, byte[] data)
			{
				return PrxTUSSetData(slotID, data, data.Length);
			}
			
			public static ErrorCode RequestTusData(int slotID)
			{
				return PrxTUSRequestData(slotID);
			}
			
			public static ErrorCode SetTusDataForUser(int slotID, byte[] data, byte[] npID)
			{
				return PrxTUSSetDataForUser(slotID, data, data.Length, npID);
			}
			
			public static ErrorCode SetTusDataForVirtualUser(int slotID, byte[] data, string onlineID)
			{
				return PrxTUSSetDataForVirtualUser(slotID, data, data.Length, onlineID);
			}

			public static ErrorCode RequestTusDataForUser(int slotID, byte[] npID)
			{
				return PrxTUSRequestDataForUser(slotID, npID);
			}
			
			public static ErrorCode RequestTusDataForVirtualUser(int slotID, string onlineID)
			{
				return PrxTUSRequestDataForVirtualUser(slotID, onlineID);
			}

			public static byte[] GetTusData()
			{
				TusTssData tusData = new TusTssData();
				if(PrxTUSGetData(out tusData) == ErrorCode.NP_OK)
				{
					return tusData.data;
				}
				return null;
			}

			public static bool IsTssBusy()
			{
				return PrxTSSIsBusy();
			}

			public static ErrorCode RequestTssData()
			{
				return PrxTSSRequestData();
			}

			//public static bool RequestTssDataFromSlot(int slot)
			//{
			//	return PrxTSSRequestDataFromSlot(slot);
			//}

			public static byte[] GetTssData()
			{
				TusTssData tusData = new TusTssData();
				if(PrxTSSGetData(out tusData) == ErrorCode.NP_OK)
				{
					return tusData.data;
				}
				return null;
			}

			// Event delegates.
			public static event Messages.EventHandler OnTusDataSet;
			public static event Messages.EventHandler OnTusDataRecieved;
			public static event Messages.EventHandler OnTusVariablesSet;
			public static event Messages.EventHandler OnTusVariablesModified;
			public static event Messages.EventHandler OnTusVariablesRecieved;
			public static event Messages.EventHandler OnTssDataRecieved;
			public static event Messages.EventHandler OnTssNoData;
			public static event Messages.EventHandler OnTusTssError;

			// Message processing.
			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
				case Messages.MessageType.kNPToolKit_TUSDataSet:
					if (OnTusDataSet != null) OnTusDataSet(msg);
					return true;

				case Messages.MessageType.kNPToolKit_TUSDataReceived:
					if (OnTusDataRecieved != null) OnTusDataRecieved(msg);
					return true;

				case Messages.MessageType.kNPToolKit_TUSVariablesSet:
					if (OnTusVariablesSet != null) OnTusVariablesSet(msg);
					return true;

				case Messages.MessageType.kNPToolKit_TUSVariablesModified:
					if (OnTusVariablesModified != null) OnTusVariablesModified(msg);
					return true;

				case Messages.MessageType.kNPToolKit_TUSVariablesReceived:
					if (OnTusVariablesRecieved != null) OnTusVariablesRecieved(msg);
					return true;

				case Messages.MessageType.kNPToolKit_TSSDataReceived:
					if (OnTssDataRecieved != null) OnTssDataRecieved(msg);
					return true;

				case Messages.MessageType.kNPToolKit_TSSNoData:
					if (OnTssNoData != null) OnTssNoData(msg);
					return true;
						
				case Messages.MessageType.kNPToolKit_TusTssError:
					if (OnTusTssError != null) OnTusTssError(msg);
					return true;
				}
				
				return false;
			}
		}
	}
}

